package gov.va.genisis2;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import java.util.Properties;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;

import gov.va.genisis2.ts.common.dto.TripleDTO;
import gov.va.genisis2.utils.TermCliConstants;

/**
 * Command Line Interface Entry Point
 * 
 * @author Michael Edoror
 *
 */
public class TermCli {

	private static final Logger LOGGER = Logger.getLogger(TermCli.class);

	public static void main(String[] args) {

		TermCliModel model = new TermCliModel();

		HttpClient httpClient = new HttpClient();

		CommandLineParser parser = new DefaultParser();
		CommandLine cmdLine = null;
		Options options = null;
		Properties properties = new Properties();
		String termSrviceUrl = null;
		
		try {
			properties.load(new FileInputStream("cli.properties"));
			termSrviceUrl = properties.getProperty("term.service.url");
			
			if (null == termSrviceUrl || termSrviceUrl.trim().length() == 0) {
				System.err.println("Missing property for Terminology Service URL (term.service.url) in cli.properties");
				System.exit(0);
			}
			
			options = createCommandLineOptions();
			cmdLine = parser.parse(options, args);
		} catch (ParseException e) {
			LOGGER.error("Couldn't parse command line arguments " + e);
			System.err.println("Error: " + e);
			HelpFormatter formatter = new HelpFormatter();
			formatter.printHelp("java -jar tsed-jar-with-dependencies.jar", options);
			System.exit(0);
		} catch (FileNotFoundException e) {
			LOGGER.error("unable to locate property file" + e);
			System.err.println("unable to locate property file: cli.properties\n" + e);
			System.exit(0);
		} catch (IOException e) {
			System.err.println("Exception " + e);
			e.printStackTrace();
		}
		
		if (cmdLine.hasOption(TermCliConstants.ADD)) {
			model.setAdd(true);
			LOGGER.debug("Adding triples");
		} else {
			// be a drop
			model.setDrop(true);
			LOGGER.debug("Dropping triples");
		}

		if (cmdLine.hasOption(TermCliConstants.CHECK_ONLY)) {
			model.setCheckOnly(true);
			LOGGER.debug("Setting check only mode true");
		}
		if (cmdLine.hasOption(TermCliConstants.HELP)) {
			HelpFormatter formatter = new HelpFormatter();
			formatter.printHelp("java -jar tsed-jar-with-dependencies.jar", options);
			System.exit(0);
		}
		if (cmdLine.hasOption(TermCliConstants.VERBOSE)) {
			model.setVerbose(true);
			LOGGER.debug("Setting check only mode true");
		}
		if (cmdLine.hasOption(TermCliConstants.IGNORE_ERR)) {
			model.setIgnoreErrors(true);
			LOGGER.debug("Setting no ignore error mode true");
		}
		if (cmdLine.hasOption(TermCliConstants.ALLOW_ERR)) {
			model.setAllowErrors(true);
			LOGGER.debug("Setting allow error mode true");
		}
		if (cmdLine.hasOption(TermCliConstants.NO_ERR)) {
			model.setNoErrors(true);
			LOGGER.debug("Setting no error mode true");
		}

		File input = null;
		if (cmdLine.hasOption(TermCliConstants.FILE)) {
			// read json from file
			input = new File(cmdLine.getOptionValue(TermCliConstants.FILE));
		} else {
			// read from standard input
			input = new File(convertStdnToFile(System.in));
		}

		ObjectMapper mapper = new ObjectMapper();
		List<TripleDTO> triples = null;

		try {
			triples = mapper.readValue(input, new TypeReference<List<TripleDTO>>() {
			});

		} catch (IOException e) {
			LOGGER.error("Error: parsing JSON" + e.getMessage());
			System.err.print("Error: parsing JSON " + e);
		}

		model.setProperties(triples);

		System.out.println(model.toJson());
		//// At this point we have the data...probably should check for issues
		//// here but now lets try to send the data
		httpClient.sendingUpdate(termSrviceUrl, model.isAdd(), model.toJson());

	}

	/**
	 * Create the option objects used as command line arguments
	 * 
	 * @return org.apache.commons.cli.Options
	 */
	private static Options createCommandLineOptions() {

		Options options = new Options();

		Option helpOption = Option.builder(TermCliConstants.HELP).required(false).desc("Displays the help message").build();

		Option fileOption = Option.builder(TermCliConstants.FILE).numberOfArgs(1).required(false).desc("Used to take in triple information from a JSON formatted file").build();

		options.addOption(helpOption);
		options.addOption(fileOption);

		// establish mutually exclusive group for required add or drop
		OptionGroup addDropGroup = new OptionGroup();
		addDropGroup.setRequired(true);

		Option add = Option.builder(TermCliConstants.ADD).desc("Triples will be added to Jena/Fuseki").build();

		Option drop = Option.builder(TermCliConstants.DROP).desc("Triples will be deleted from Jena/Fuseki").build();

		addDropGroup.addOption(add);
		addDropGroup.addOption(drop);

		options.addOptionGroup(addDropGroup);

		// establish mutually exclusive group for errors
		OptionGroup errorGroup = new OptionGroup();
		errorGroup.setRequired(true);

		Option checkOnlyOption = Option.builder("c").longOpt(TermCliConstants.CHECK_ONLY).required(false).desc("Checks for errors but performs no database updates").build();

		Option noErrorOption = Option.builder("noErr").longOpt(TermCliConstants.NO_ERR).required(false).desc("Performs updates if and only if there are no errors").build();

		Option allowErrorOption = Option.builder("allowErr").longOpt(TermCliConstants.ALLOW_ERR).required(false).desc("Performs any and all updates that have no errors").build();

		Option ignoreErrorOption = Option.builder("ignoreErr").longOpt(TermCliConstants.IGNORE_ERR).required(false).desc("Performs updates with no discretion.  Default mode").build();

		errorGroup.addOption(checkOnlyOption);
		errorGroup.addOption(noErrorOption);
		errorGroup.addOption(allowErrorOption);
		errorGroup.addOption(ignoreErrorOption);

		options.addOptionGroup(errorGroup);

		return options;

	}

	private static String convertStdnToFile(InputStream in) {

		try (OutputStream outputStream = new FileOutputStream(new File(TermCliConstants.TEMP_INPUT_FILE))) {
			int read = 0;
			byte[] bytes = new byte[1024];

			while ((read = in.read(bytes)) != -1) {
				outputStream.write(bytes, 0, read);
			}
		} catch (IOException e) {
			LOGGER.error("Error converting stdin to file: " + e.getMessage());
		} finally {
			if (in != null)
				try {
					in.close();
				} catch (IOException e) {
					LOGGER.error("Error couldn't close file: " + e.getMessage());
				}
		}
		return TermCliConstants.TEMP_INPUT_FILE;
	}
}
